import warnings
import pickle
from pathlib import Path
import yaml
import functools

import jax
import jax.numpy as jnp
from flax import serialization
import pandas as pd

import dreamerv3
from dreamerv3 import embodied
from dreamerv3.agent import ImagActorCritic
from dreamerv3.embodied.core.goal_sampler import GoalSampler, GoalSamplerCyclic
from baselines.qdax.core.containers.mapelites_repertoire import compute_euclidean_centroids
from analysis.data_utils.analysis_repertoire import AnalysisRepertoire, AnalysisLatentRepertoire
from baselines.qdax import environments
from baselines.qdax.core.neuroevolution.networks.networks import MLPDC
from baselines.qdax.baselines.diayn_smerl import DIAYNSMERL, DiaynSmerlConfig
from baselines.qdax.core.neuroevolution.buffers.buffer import QDTransition
from baselines.qdax.tasks.brax_envs import reset_based_scoring_actor_dc_function_brax_envs as scoring_actor_dc_function
from baselines.qdax.environments import get_feat_mean
from baselines.qdax.utils.plotting import plot_2d_map_elites_repertoire

from omegaconf import OmegaConf
from utils import get_env
warnings.filterwarnings("ignore", ".*truncated to dtype int32.*")


task = "humanoid"

ours_path = Path("/project/output/results/ours/humanoid_jump")
smerl_path = Path("/project/output/results/smerl/humanoid_jump")
smerl_reverse_path = Path("/project/output/results/smerl_reverse/humanoid_jump")
uvfa_path = Path("/project/output/results/uvfa/humanoid_jump")
dcg_me_path = Path("/project/output/results/dcg_me/humanoid_jump")


def eval_ours(run_path, index, random_key):
    try:
        config_path = list((run_path / "wandb").iterdir())[0] / "files" / "config.yaml"
        with open(config_path) as f:
            config = yaml.safe_load(f)
    except:
        config_path = run_path / "wandb" / "latest-run" / "files" / "config.yaml"
        with open(config_path) as f:
            config = yaml.safe_load(f)

    argv = [
    "--task={}".format(config["task"]["value"]),
    "--feat={}".format(config["feat"]["value"]),
    "--backend={}".format(config["backend"]["value"]),
    "--goal.resolution={}".format(config["goal"]["value"]["resolution"]),

    "--run.from_checkpoint={}".format(str(run_path / "checkpoint.ckpt")),
    "--envs.amount=50",
    ]

    # Create config
    logdir = str(run_path)
    config = embodied.Config(dreamerv3.configs["defaults"])
    config = config.update(dreamerv3.configs["brax"])
    config = config.update({
    "logdir": logdir,
    "run.train_ratio": 32,
    "run.log_every": 60,  # Seconds
    "batch_size": 16,
    })
    config = embodied.Flags(config).parse(argv=argv)

    # Create logger
    logdir = embodied.Path(config.logdir)
    step = embodied.Counter()
    logger = embodied.Logger(step, [
    embodied.logger.TerminalOutput(),
    embodied.logger.JSONLOutput(logdir, "metrics.jsonl"),
    embodied.logger.TensorBoardOutput(logdir),
    # embodied.logger.WandBOutput(logdir, config),
    # embodied.logger.MLFlowOutput(logdir.name),
    ])

    # Create environment
    # env = get_env(config, mode="train", index=index)
    env = get_env(config, mode="train", index=index)

    # Create agent and replay buffer
    agent = dreamerv3.Agent(env.obs_space, env.act_space, env.feat_space, step, config)
    args = embodied.Config(
        **config.run, logdir=config.logdir,
        batch_steps=config.batch_size * config.batch_length)

    # Create goal sampler
    resolution = ImagActorCritic.get_resolution(env.feat_space, config)
    grid_shape = (resolution,) * env.feat_space['vector'].shape[0]
    goals = compute_euclidean_centroids(grid_shape, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    goal_sampler_cyclic = GoalSamplerCyclic(feat_space=env.feat_space, 
                                            goal_list=list(goals),
                                            number_visits_per_goal=n_visits_per_goal)
    embodied.run.eval_only(agent,
                           env,
                           goal_sampler=goal_sampler_cyclic,
                           period_sample_goals=float('inf'),
                           logger=logger,
                           args=args,)

    ours_repertoire = AnalysisRepertoire.create_from_path_collection_results(run_path / "results_dreamer.pkl")
    # plot_repertoire = ours_repertoire.replace(descriptors=jnp.mean(ours_repertoire.descriptors, axis=1), fitnesses=jnp.mean(ours_repertoire.fitnesses, axis=1))
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, plot_repertoire.fitnesses, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/ours_fitness.png")
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, -jnp.linalg.norm(plot_repertoire.centroids-plot_repertoire.descriptors, axis=-1), minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/ours_distance_to_goal.png")
    return ours_repertoire

def eval_smerl(run_path, index, random_key):
    with open(run_path / ".hydra" / "config.yaml") as f:
        config = yaml.safe_load(f)
    config = OmegaConf.load(run_path / ".hydra" / "config.yaml")

    # Init environment
    env = environments.create(config.task + "hurdle" + "_" + config.feat,
                              episode_length=config.algo.episode_length,
                              backend=config.algo.backend,
                              index=index,)

    # Define config
    smerl_config = DiaynSmerlConfig(
        # SAC config
        batch_size=config.algo.batch_size,
        episode_length=config.algo.episode_length,
        tau=config.algo.soft_tau_update,
        normalize_observations=config.algo.normalize_observations,
        learning_rate=config.algo.learning_rate,
        alpha_init=config.algo.alpha_init,
        discount=config.algo.discount,
        reward_scaling=config.algo.reward_scaling,
        hidden_layer_sizes=config.algo.hidden_layer_sizes,
        fix_alpha=config.algo.fix_alpha,
        # DIAYN config
        skill_type=config.algo.skill_type,
        num_skills=config.algo.num_skills,
        descriptor_full_state=config.algo.descriptor_full_state,
        extrinsic_reward=False,
        beta=1.,
        # SMERL
        reverse=False,
        diversity_reward_scale=config.algo.diversity_reward_scale,
        smerl_target=config.algo.smerl_target,
        smerl_margin=config.algo.smerl_margin,
    )

    # Define an instance of DIAYN
    smerl = DIAYNSMERL(config=smerl_config, action_size=env.action_size)

    random_key, random_subkey_1, random_subkey_2 = jax.random.split(random_key, 3)
    fake_obs = jnp.zeros((env.observation_size + config.algo.num_skills,))
    fake_goal = jnp.zeros((config.algo.num_skills,))
    fake_actor_params = smerl._policy.init(random_subkey_1, fake_obs)
    fake_discriminator_params = smerl._discriminator.init(random_subkey_2, fake_goal)

    with open(run_path / "actor/actor.pickle", "rb") as params_file:
        state_dict = pickle.load(params_file)
    actor_params = serialization.from_state_dict(fake_actor_params, state_dict)

    with open(run_path / "discriminator/discriminator.pickle", "rb") as params_file:
        state_dict = pickle.load(params_file)
    discriminator_params = serialization.from_state_dict(fake_discriminator_params, state_dict)

    # Create grid
    resolution = 50
    grid_shape = (resolution,) * env.feat_space['vector'].shape[0]
    goals = compute_euclidean_centroids(grid_shape, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    latent_goals, _ = smerl._discriminator.apply(discriminator_params, goals)

    reset_fn = jax.jit(env.reset)

    @jax.jit
    def play_step_fn(env_state, params, latent_goal, random_key):
        actions, random_key = smerl.select_action(
                    obs=jnp.concatenate([env_state.obs, latent_goal], axis=0),
                    policy_params=params,
                    random_key=random_key,
                    deterministic=True,
                )
        state_desc = env_state.info["state_descriptor"]
        next_state = env.step(env_state, actions)

        transition = QDTransition(
            obs=env_state.obs,
            next_obs=next_state.obs,
            rewards=next_state.reward,
            dones=next_state.done,
            truncations=next_state.info["truncation"],
            actions=actions,
            state_desc=state_desc,
            next_state_desc=next_state.info["state_descriptor"],
            desc=jnp.zeros(env.behavior_descriptor_length,) * jnp.nan,
            desc_prime=jnp.zeros(env.behavior_descriptor_length,) * jnp.nan,
        )

        return next_state, params, latent_goal, random_key, transition

    # Prepare the scoring function
    scoring_fn = jax.jit(functools.partial(
        scoring_actor_dc_function,
        episode_length=config.algo.episode_length,
        play_reset_fn=reset_fn,
        play_step_actor_dc_fn=play_step_fn,
        behavior_descriptor_extractor=get_feat_mean,
    ))

    @jax.jit
    def evaluate_actor(random_key, params, latent_goals):
        params = jax.tree_util.tree_map(lambda x: jnp.repeat(jnp.expand_dims(x, axis=0), latent_goals.shape[0], axis=0), params)
        fitnesses, descriptors, extra_scores, random_key = scoring_fn(
            params, latent_goals, random_key
        )
        return fitnesses, descriptors, extra_scores, random_key
    
    fitnesses_list = []
    descriptor_list = []
    for _ in range(n_visits_per_goal):
        fitnesses, descriptors, extra_scores, random_key = evaluate_actor(random_key, actor_params, latent_goals)
        fitnesses_list.append(fitnesses)
        descriptor_list.append(descriptors)

    smerl_repertoire = AnalysisLatentRepertoire(
        centroids=goals,
        latent_goals=latent_goals,
        fitnesses=jnp.stack(fitnesses_list, axis=1),
        descriptors=jnp.stack(descriptor_list, axis=1))
    # plot_repertoire = smerl_repertoire.replace(descriptors=jnp.mean(smerl_repertoire.descriptors, axis=1), fitnesses=jnp.mean(smerl_repertoire.fitnesses, axis=1))
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, plot_repertoire.fitnesses, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/smerl_fitness.png")
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, -jnp.linalg.norm(goals-descriptors, axis=-1), minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/smerl_distance_to_goal.png")
    return smerl_repertoire

def eval_smerl_reverse(run_path, index, random_key):
    with open(run_path / ".hydra" / "config.yaml") as f:
        config = yaml.safe_load(f)
    config = OmegaConf.load(run_path / ".hydra" / "config.yaml")

    # Init environment
    env = environments.create(config.task + "hurdle" + "_" + config.feat,
                              episode_length=config.algo.episode_length,
                              backend=config.algo.backend,
                              index=index,)

    # Define config
    smerl_config = DiaynSmerlConfig(
        # SAC config
        batch_size=config.algo.batch_size,
        episode_length=config.algo.episode_length,
        tau=config.algo.soft_tau_update,
        normalize_observations=config.algo.normalize_observations,
        learning_rate=config.algo.learning_rate,
        alpha_init=config.algo.alpha_init,
        discount=config.algo.discount,
        reward_scaling=config.algo.reward_scaling,
        hidden_layer_sizes=config.algo.hidden_layer_sizes,
        fix_alpha=config.algo.fix_alpha,
        # DIAYN config
        skill_type=config.algo.skill_type,
        num_skills=config.algo.num_skills,
        descriptor_full_state=config.algo.descriptor_full_state,
        extrinsic_reward=False,
        beta=1.,
        # SMERL
        reverse=True,
        diversity_reward_scale=config.algo.diversity_reward_scale,
        smerl_target=config.algo.smerl_target,
        smerl_margin=config.algo.smerl_margin,
    )

    # Define an instance of DIAYN
    smerl = DIAYNSMERL(config=smerl_config, action_size=env.action_size)

    random_key, random_subkey_1, random_subkey_2 = jax.random.split(random_key, 3)
    fake_obs = jnp.zeros((env.observation_size + config.algo.num_skills,))
    fake_goal = jnp.zeros((config.algo.num_skills,))
    fake_actor_params = smerl._policy.init(random_subkey_1, fake_obs)
    fake_discriminator_params = smerl._discriminator.init(random_subkey_2, fake_goal)

    with open(run_path / "actor/actor.pickle", "rb") as params_file:
        state_dict = pickle.load(params_file)
    actor_params = serialization.from_state_dict(fake_actor_params, state_dict)

    with open(run_path / "discriminator/discriminator.pickle", "rb") as params_file:
        state_dict = pickle.load(params_file)
    discriminator_params = serialization.from_state_dict(fake_discriminator_params, state_dict)

    # Create grid
    resolution = 50
    grid_shape = (resolution,) * env.feat_space['vector'].shape[0]
    goals = compute_euclidean_centroids(grid_shape, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    latent_goals, _ = smerl._discriminator.apply(discriminator_params, goals)

    reset_fn = jax.jit(env.reset)

    @jax.jit
    def play_step_fn(env_state, params, latent_goal, random_key):
        actions, random_key = smerl.select_action(
                    obs=jnp.concatenate([env_state.obs, latent_goal], axis=0),
                    policy_params=params,
                    random_key=random_key,
                    deterministic=True,
                )
        state_desc = env_state.info["state_descriptor"]
        next_state = env.step(env_state, actions)

        transition = QDTransition(
            obs=env_state.obs,
            next_obs=next_state.obs,
            rewards=next_state.reward,
            dones=next_state.done,
            truncations=next_state.info["truncation"],
            actions=actions,
            state_desc=state_desc,
            next_state_desc=next_state.info["state_descriptor"],
            desc=jnp.zeros(env.behavior_descriptor_length,) * jnp.nan,
            desc_prime=jnp.zeros(env.behavior_descriptor_length,) * jnp.nan,
        )

        return next_state, params, latent_goal, random_key, transition

    # Prepare the scoring function
    scoring_fn = jax.jit(functools.partial(
        scoring_actor_dc_function,
        episode_length=config.algo.episode_length,
        play_reset_fn=reset_fn,
        play_step_actor_dc_fn=play_step_fn,
        behavior_descriptor_extractor=get_feat_mean,
    ))

    @jax.jit
    def evaluate_actor(random_key, params, latent_goals):
        params = jax.tree_util.tree_map(lambda x: jnp.repeat(jnp.expand_dims(x, axis=0), latent_goals.shape[0], axis=0), params)
        fitnesses, descriptors, extra_scores, random_key = scoring_fn(
            params, latent_goals, random_key
        )
        return fitnesses, descriptors, extra_scores, random_key
    
    fitnesses_list = []
    descriptor_list = []
    for _ in range(n_visits_per_goal):
        fitnesses, descriptors, extra_scores, random_key = evaluate_actor(random_key, actor_params, latent_goals)
        fitnesses_list.append(fitnesses)
        descriptor_list.append(descriptors)

    smerl_repertoire = AnalysisLatentRepertoire(
        centroids=goals,
        latent_goals=latent_goals,
        fitnesses=jnp.stack(fitnesses_list, axis=1),
        descriptors=jnp.stack(descriptor_list, axis=1))
    # plot_repertoire = smerl_repertoire.replace(descriptors=jnp.mean(smerl_repertoire.descriptors, axis=1), fitnesses=jnp.mean(smerl_repertoire.fitnesses, axis=1))
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, plot_repertoire.fitnesses, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/smerl_reverse_fitness.png")
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, -jnp.linalg.norm(goals-descriptors, axis=-1), minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/smerl_reverse_distance_to_goal.png")
    return smerl_repertoire

def eval_uvfa(run_path, index, random_key):
    config_path = run_path / "wandb" / "latest-run" / "files" / "config.yaml"
    with open(config_path) as f:
        config = yaml.safe_load(f)

    argv = [
    "--task={}".format(config["task"]["value"]),
    "--feat={}".format(config["feat"]["value"]),
    "--backend={}".format(config["backend"]["value"]),

    "--run.from_checkpoint={}".format(str(run_path / "checkpoint.ckpt")),
    "--envs.amount=50",
    ]

    # Create config
    logdir = str(run_path)
    config = embodied.Config(dreamerv3.configs["defaults"])
    config = config.update(dreamerv3.configs["brax"])
    config = config.update({
    "logdir": logdir,
    "run.train_ratio": 32,
    "run.log_every": 60,  # Seconds
    "batch_size": 16,
    })
    config = embodied.Flags(config).parse(argv=argv)

    # Create logger
    logdir = embodied.Path(config.logdir)
    step = embodied.Counter()
    logger = embodied.Logger(step, [
    embodied.logger.TerminalOutput(),
    embodied.logger.JSONLOutput(logdir, "metrics.jsonl"),
    embodied.logger.TensorBoardOutput(logdir),
    # embodied.logger.WandBOutput(logdir, config),
    # embodied.logger.MLFlowOutput(logdir.name),
    ])

    # Create environment
    env = get_env(config, mode="train", index=index)

    # Create agent and replay buffer
    agent = dreamerv3.Agent(env.obs_space, env.act_space, env.feat_space, step, config)
    args = embodied.Config(
        **config.run, logdir=config.logdir,
        batch_steps=config.batch_size * config.batch_length)

    # Create goal sampler
    resolution = ImagActorCritic.get_resolution(env.feat_space, config)
    grid_shape = (resolution,) * env.feat_space['vector'].shape[0]
    goals = compute_euclidean_centroids(grid_shape, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    goal_sampler_cyclic = GoalSamplerCyclic(feat_space=env.feat_space, 
                                            goal_list=list(goals),
                                            number_visits_per_goal=n_visits_per_goal)
    embodied.run.eval_only(agent,
                           env,
                           goal_sampler=goal_sampler_cyclic,
                           period_sample_goals=float('inf'),
                           logger=logger,
                           args=args,)

    ours_repertoire = AnalysisRepertoire.create_from_path_collection_results(run_path / "results_dreamer.pkl")
    # plot_repertoire = ours_repertoire.replace(descriptors=jnp.mean(ours_repertoire.descriptors, axis=1), fitnesses=jnp.mean(ours_repertoire.fitnesses, axis=1))
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, plot_repertoire.fitnesses, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/ours_fitness.png")
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, -jnp.linalg.norm(plot_repertoire.centroids-plot_repertoire.descriptors, axis=-1), minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/ours_distance_to_goal.png")
    return ours_repertoire

def eval_dcg_me(run_path, index, random_key):
    with open(run_path / ".hydra" / "config.yaml") as f:
        config = yaml.safe_load(f)
    config = OmegaConf.load(run_path / ".hydra" / "config.yaml")

    # Init environment
    env = environments.create(config.task + "hurdle" + "_" + config.feat,
                              episode_length=config.algo.episode_length,
                              backend=config.algo.backend,
                              index=index,)
    reset_fn = jax.jit(env.reset)

    # Init policy network
    policy_layer_sizes = config.algo.policy_hidden_layer_sizes + (env.action_size,)
    actor_dc_network = MLPDC(
        layer_sizes=policy_layer_sizes,
        kernel_init=jax.nn.initializers.lecun_uniform(),
        final_activation=jnp.tanh,
    )

    # Init population of controllers
    random_key, subkey = jax.random.split(random_key)
    fake_obs = jnp.zeros(shape=(env.observation_size,))
    fake_desc = jnp.zeros(shape=(env.behavior_descriptor_length,))
    fake_actor_params = actor_dc_network.init(subkey, fake_obs, fake_desc)

    with open(run_path / "actor/actor.pickle", "rb") as params_file:
        state_dict = pickle.load(params_file)
    actor_params = serialization.from_state_dict(fake_actor_params, state_dict)

    # Create grid
    resolution = 50
    grid_shape = (resolution,) * env.feat_space['vector'].shape[0]
    goals = compute_euclidean_centroids(grid_shape, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)

    def play_step_actor_dc_fn(env_state, actor_dc_params, desc, random_key):
        actions = actor_dc_network.apply(actor_dc_params, env_state.obs, desc/env.behavior_descriptor_limits[1][0])
        state_desc = env_state.info["feat"]
        next_state = env.step(env_state, actions)

        transition = QDTransition(
            obs=env_state.obs,
            next_obs=next_state.obs,
            rewards=next_state.reward,
            dones=next_state.done,
            truncations=next_state.info["truncation"],
            actions=actions,
            state_desc=state_desc,
            next_state_desc=next_state.info["feat"],
            desc=jnp.zeros(env.behavior_descriptor_length,) * jnp.nan,
            desc_prime=desc/env.behavior_descriptor_limits[1][0],
        )

        return next_state, actor_dc_params, desc, random_key, transition

    # Prepare the scoring function
    scoring_fn = jax.jit(functools.partial(
        scoring_actor_dc_function,
        episode_length=env.episode_length,
        play_reset_fn=reset_fn,
        play_step_actor_dc_fn=play_step_actor_dc_fn,
        behavior_descriptor_extractor=get_feat_mean,
    ))

    @jax.jit
    def evaluate_actor(random_key, params, goals):
        params = jax.tree_util.tree_map(lambda x: jnp.repeat(jnp.expand_dims(x, axis=0), goals.shape[0], axis=0), params)
        fitnesses, descriptors, extra_scores, random_key = scoring_fn(
            params, goals, random_key
        )
        return fitnesses, descriptors, extra_scores, random_key
    
    fitnesses_list = []
    descriptor_list = []
    for _ in range(n_visits_per_goal):
        fitnesses, descriptors, extra_scores, random_key = evaluate_actor(random_key, actor_params, goals)
        fitnesses_list.append(fitnesses)
        descriptor_list.append(descriptors)

    smerl_repertoire = AnalysisRepertoire(
        centroids=goals,
        fitnesses=jnp.stack(fitnesses_list, axis=1),
        descriptors=jnp.stack(descriptor_list, axis=1))
    # plot_repertoire = smerl_repertoire.replace(descriptors=jnp.mean(smerl_repertoire.descriptors, axis=1), fitnesses=jnp.mean(smerl_repertoire.fitnesses, axis=1))
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, plot_repertoire.fitnesses, minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/dcg_me_fitness.png")
    # fig, _ = plot_2d_map_elites_repertoire(plot_repertoire.centroids, -jnp.linalg.norm(goals-descriptors, axis=-1), minval=env.feat_space['vector'].low, maxval=env.feat_space['vector'].high)
    # fig.savefig("/project/output/hierarchy/dcg_me_distance_to_goal.png")
    return smerl_repertoire


n_seeds = 10
n_actuator_failure = 19
n_visits_per_goal = 1

ours_seeds = list(ours_path.iterdir())
smerl_seeds = list(smerl_path.iterdir())
smerl_reverse_seeds = list(smerl_reverse_path.iterdir())
uvfa_seeds = list(uvfa_path.iterdir())
dcg_me_seeds = list(dcg_me_path.iterdir())

random_key = jax.random.PRNGKey(1234)

df = pd.DataFrame(columns=["algo", "seed", "height", "fitness", "distance_to_goal"])
for i in range(n_seeds):
    if i <= 6:
        continue
    for j, height in enumerate(jnp.linspace(0, 0.5, 20)):
        if j == 0:
            continue
        if i == 7 and j <= 5:
            continue
        print(f"seed: {i}, {j}/{n_actuator_failure}")
        # ours
        print(f"ours: {height}")
        random_key, subkey = jax.random.split(random_key)
        ours_repertoire = eval_ours(ours_seeds[i], j, subkey)
        df.loc[len(df)] = ["ours", i, height, jnp.max(jnp.median(ours_repertoire.fitnesses, axis=-1)), jnp.mean(-jnp.linalg.norm(ours_repertoire.centroids - jnp.mean(ours_repertoire.descriptors, axis=1), axis=-1))]

        # smerl
        print(f"smerl: {height}")
        random_key, subkey = jax.random.split(random_key)
        smerl_repertoire = eval_smerl(smerl_seeds[i], j, subkey)
        df.loc[len(df)] = ["smerl", i, height, jnp.max(jnp.median(smerl_repertoire.fitnesses, axis=-1)), jnp.mean(-jnp.linalg.norm(smerl_repertoire.centroids - jnp.mean(smerl_repertoire.descriptors, axis=1), axis=-1))]

        # smerl_reverse
        print(f"smerl_reverse: {height}")
        random_key, subkey = jax.random.split(random_key)
        smerl_reverse_repertoire = eval_smerl_reverse(smerl_reverse_seeds[i], j, subkey)
        df.loc[len(df)] = ["smerl_reverse", i, height, jnp.max(jnp.median(smerl_reverse_repertoire.fitnesses, axis=-1)), jnp.mean(-jnp.linalg.norm(smerl_reverse_repertoire.centroids - jnp.mean(smerl_reverse_repertoire.descriptors, axis=1), axis=-1))]

        # ours
        print(f"uvfa: {height}")
        random_key, subkey = jax.random.split(random_key)
        uvfa_repertoire = eval_uvfa(uvfa_seeds[i], j, subkey)
        df.loc[len(df)] = ["uvfa", i, height, jnp.max(jnp.median(uvfa_repertoire.fitnesses, axis=-1)), jnp.mean(-jnp.linalg.norm(uvfa_repertoire.centroids - jnp.mean(uvfa_repertoire.descriptors, axis=1), axis=-1))]

        # dcg_me
        print(f"dcg_me: {height}")
        random_key, subkey = jax.random.split(random_key)
        dcg_me_repertoire = eval_dcg_me(dcg_me_seeds[i], j, subkey)
        df.loc[len(df)] = ["dcg_me", i, height, jnp.max(jnp.median(dcg_me_repertoire.fitnesses, axis=-1)), jnp.mean(-jnp.linalg.norm(dcg_me_repertoire.centroids - jnp.mean(dcg_me_repertoire.descriptors, axis=1), axis=-1))]

        df.to_csv(f"/project/output/hierarchy/humanoidhurdle_{n_seeds}_{n_actuator_failure}_{n_visits_per_goal}.csv")
